/*
 * Decompiled with CFR 0.152.
 */
package com.sistema.turistico.service;

import com.sistema.turistico.dto.MovimientoCajaRequest;
import com.sistema.turistico.dto.PagoReservaRequest;
import com.sistema.turistico.entity.Caja;
import com.sistema.turistico.entity.MovimientoCaja;
import com.sistema.turistico.entity.PagoReserva;
import com.sistema.turistico.entity.Reserva;
import com.sistema.turistico.entity.Usuario;
import com.sistema.turistico.repository.PagoReservaRepository;
import com.sistema.turistico.repository.UsuarioRepository;
import com.sistema.turistico.security.TenantContext;
import com.sistema.turistico.service.CajaService;
import com.sistema.turistico.service.ReservaService;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
@Transactional
public class PagoReservaService {
    private static final Logger log = LoggerFactory.getLogger(PagoReservaService.class);
    private static final Set<String> METODOS_VALIDOS = Set.of("Efectivo", "Tarjeta Cr\u00e9dito", "Tarjeta D\u00e9bito", "Transferencia", "Yape/Plin", "Otros");
    private final PagoReservaRepository pagoReservaRepository;
    private final ReservaService reservaService;
    private final CajaService cajaService;
    private final UsuarioRepository usuarioRepository;

    public PagoReserva registrarPago(Long reservaId, PagoReservaRequest request) {
        if (request == null) {
            throw new IllegalArgumentException("La solicitud es obligatoria");
        }
        log.info("Registrando pago para reserva {} por usuario {}", (Object)reservaId, (Object)request.getUsuarioId());
        if (request.getCajaId() == null) {
            throw new IllegalArgumentException("La caja es obligatoria");
        }
        if (request.getMontoPagado() == null) {
            throw new IllegalArgumentException("El monto pagado es obligatorio");
        }
        this.validarMetodoPago(request.getMetodoPago());
        Reserva reserva = this.obtenerReservaAutorizada(reservaId);
        Long empresaId = reserva.getEmpresa() != null ? reserva.getEmpresa().getIdEmpresa() : null;
        Usuario usuario = this.obtenerUsuarioAutorizado(request.getUsuarioId(), empresaId);
        Caja caja = this.cajaService.obtenerCajaActiva(request.getCajaId());
        this.validarMismaEmpresa(empresaId, caja.getEmpresa() != null ? caja.getEmpresa().getIdEmpresa() : null, "caja");
        BigDecimal totalReserva = reserva.getPrecioFinal();
        BigDecimal acumuladoAnterior = this.pagoReservaRepository.sumMontosActivosPorReserva(reservaId);
        BigDecimal saldoPendiente = totalReserva.subtract(acumuladoAnterior);
        if (request.getMontoPagado().compareTo(saldoPendiente) > 0) {
            throw new IllegalArgumentException("El monto excede el saldo pendiente de la reserva");
        }
        PagoReserva pago = new PagoReserva();
        pago.setReserva(reserva);
        pago.setUsuario(usuario);
        pago.setMontoPagado(request.getMontoPagado());
        pago.setMetodoPago(this.normalizarMetodoPago(request.getMetodoPago()));
        pago.setNumeroOperacion(this.normalizarTexto(request.getNumeroOperacion()));
        pago.setComprobante(this.normalizarTexto(request.getComprobante()));
        pago.setFechaPago(request.getFechaPago() != null ? request.getFechaPago() : LocalDate.now());
        pago.setObservaciones(this.normalizarTexto(request.getObservaciones()));
        pago.setEstado(Boolean.TRUE);
        PagoReserva pagoGuardado = (PagoReserva)this.pagoReservaRepository.save((Object)pago);
        MovimientoCajaRequest movimientoRequest = new MovimientoCajaRequest();
        movimientoRequest.setTipoMovimiento(MovimientoCaja.TipoMovimiento.Ingreso);
        movimientoRequest.setMonto(pagoGuardado.getMontoPagado());
        movimientoRequest.setDescripcion("Ingreso por pago de reserva " + reserva.getCodigoReserva());
        this.cajaService.registrarMovimiento(caja.getIdCaja(), movimientoRequest);
        BigDecimal totalPagado = acumuladoAnterior.add(pagoGuardado.getMontoPagado());
        Reserva.EstadoReserva nuevoEstado = this.determinarEstadoFinanciero(totalReserva, totalPagado);
        this.reservaService.actualizarEstadoFinanciero(reserva, nuevoEstado);
        return pagoGuardado;
    }

    @Transactional(readOnly=true)
    public List<PagoReserva> listarPagosReserva(Long reservaId) {
        Reserva reserva = this.obtenerReservaAutorizada(reservaId);
        return this.pagoReservaRepository.findByReserva_IdReservaOrderByFechaPagoAsc(reserva.getIdReserva());
    }

    @Transactional(readOnly=true)
    public List<PagoReserva> listarPagosActivosReserva(Long reservaId) {
        Reserva reserva = this.obtenerReservaAutorizada(reservaId);
        return this.pagoReservaRepository.findByReserva_IdReservaAndEstadoTrueOrderByFechaPagoAsc(reserva.getIdReserva());
    }

    @Transactional(readOnly=true)
    public List<PagoReserva> listarPagosPorFiltros(Long empresaId, String metodoPago, Boolean estado, LocalDate fechaDesde, LocalDate fechaHasta) {
        String metodoNormalizado = null;
        if (metodoPago != null) {
            this.validarMetodoPago(metodoPago);
            metodoNormalizado = this.normalizarMetodoPago(metodoPago);
        }
        if (fechaDesde != null && fechaHasta != null && fechaHasta.isBefore(fechaDesde)) {
            throw new IllegalArgumentException("El rango de fechas es inv\u00e1lido");
        }
        Long empresaFiltrada = TenantContext.resolveEmpresaId((Long)empresaId);
        return this.pagoReservaRepository.findByFiltros(empresaFiltrada, metodoNormalizado, estado, fechaDesde, fechaHasta).stream().filter(arg_0 -> this.puedeVerPago(arg_0)).toList();
    }

    public PagoReserva anularPago(Long reservaId, Long pagoId, Long cajaId, String motivo) {
        log.info("Anulando pago {} de reserva {}", (Object)pagoId, (Object)reservaId);
        if (cajaId == null) {
            throw new IllegalArgumentException("La caja es obligatoria");
        }
        Reserva reserva = this.obtenerReservaAutorizada(reservaId);
        Caja caja = this.cajaService.obtenerCajaActiva(cajaId);
        this.validarMismaEmpresa(reserva.getEmpresa() != null ? reserva.getEmpresa().getIdEmpresa() : null, caja.getEmpresa() != null ? caja.getEmpresa().getIdEmpresa() : null, "caja");
        PagoReserva pago = (PagoReserva)this.pagoReservaRepository.findByIdPagoAndReserva_IdReservaAndEstadoTrue(pagoId, reserva.getIdReserva()).orElseThrow(() -> new IllegalArgumentException("El pago indicado no est\u00e1 activo"));
        this.validarPertenencia(pago);
        pago.setEstado(Boolean.FALSE);
        String motivoNormalizado = this.normalizarTexto(motivo);
        if (motivoNormalizado != null) {
            String observaciones = pago.getObservaciones();
            String anotacion = "Anulado: " + motivoNormalizado + " (" + String.valueOf(LocalDateTime.now()) + ")";
            pago.setObservaciones(observaciones == null ? anotacion : observaciones + " | " + anotacion);
        }
        PagoReserva pagoAnulado = (PagoReserva)this.pagoReservaRepository.save((Object)pago);
        MovimientoCajaRequest movimientoRequest = new MovimientoCajaRequest();
        movimientoRequest.setTipoMovimiento(MovimientoCaja.TipoMovimiento.Egreso);
        movimientoRequest.setMonto(pagoAnulado.getMontoPagado());
        movimientoRequest.setDescripcion("Reverso pago reserva " + pago.getReserva().getCodigoReserva());
        this.cajaService.registrarMovimiento(caja.getIdCaja(), movimientoRequest);
        BigDecimal totalPagado = this.pagoReservaRepository.sumMontosActivosPorReserva(reserva.getIdReserva());
        BigDecimal totalReserva = pago.getReserva().getPrecioFinal();
        Reserva.EstadoReserva nuevoEstado = this.determinarEstadoFinanciero(totalReserva, totalPagado);
        this.reservaService.actualizarEstadoFinanciero(pago.getReserva(), nuevoEstado);
        return pagoAnulado;
    }

    @Transactional(readOnly=true)
    public PagoReserva obtenerPago(Long pagoId) {
        return this.obtenerPagoAutorizado(pagoId);
    }

    public PagoReserva actualizarPago(Long pagoId, PagoReservaRequest request) {
        log.info("Actualizando pago {}", (Object)pagoId);
        PagoReserva pago = this.obtenerPagoAutorizado(pagoId);
        if (request == null) {
            throw new IllegalArgumentException("La solicitud es obligatoria");
        }
        if (request.getMetodoPago() != null) {
            this.validarMetodoPago(request.getMetodoPago());
            pago.setMetodoPago(this.normalizarMetodoPago(request.getMetodoPago()));
        }
        if (request.getNumeroOperacion() != null) {
            pago.setNumeroOperacion(this.normalizarTexto(request.getNumeroOperacion()));
        }
        if (request.getComprobante() != null) {
            pago.setComprobante(this.normalizarTexto(request.getComprobante()));
        }
        if (request.getFechaPago() != null) {
            pago.setFechaPago(request.getFechaPago());
        }
        if (request.getObservaciones() != null) {
            pago.setObservaciones(this.normalizarTexto(request.getObservaciones()));
        }
        return (PagoReserva)this.pagoReservaRepository.save((Object)pago);
    }

    @Transactional(readOnly=true)
    public BigDecimal obtenerSaldoPendiente(Long reservaId) {
        Reserva reserva = this.obtenerReservaAutorizada(reservaId);
        BigDecimal pagado = this.pagoReservaRepository.sumMontosActivosPorReserva(reservaId);
        return reserva.getPrecioFinal().subtract(pagado);
    }

    private Reserva obtenerReservaAutorizada(Long reservaId) {
        return (Reserva)this.reservaService.findById(reservaId).orElseThrow(() -> new IllegalArgumentException("Reserva no encontrada"));
    }

    private PagoReserva obtenerPagoAutorizado(Long pagoId) {
        PagoReserva pago = (PagoReserva)this.pagoReservaRepository.findById((Object)pagoId).orElseThrow(() -> new IllegalArgumentException("Pago no encontrado"));
        this.validarPertenencia(pago);
        return pago;
    }

    private Usuario obtenerUsuarioAutorizado(Long usuarioId, Long empresaEsperada) {
        Long empresaActual;
        Long resolvedId = usuarioId != null ? usuarioId : TenantContext.requireUserId();
        Usuario usuario = (Usuario)this.usuarioRepository.findActivoById(resolvedId).orElseThrow(() -> new IllegalArgumentException("Usuario no encontrado"));
        Long empresaUsuario = usuario.getEmpresa() != null ? usuario.getEmpresa().getIdEmpresa() : null;
        this.validarMismaEmpresa(empresaEsperada, empresaUsuario, "usuario");
        if (!this.esSuperAdmin() && !(empresaActual = TenantContext.requireEmpresaId()).equals(empresaUsuario)) {
            throw new IllegalArgumentException("El usuario no pertenece a la empresa actual");
        }
        return usuario;
    }

    private void validarPertenencia(PagoReserva pago) {
        if (!this.esSuperAdmin()) {
            Long empresaPago;
            Long empresaActual = TenantContext.requireEmpresaId();
            Long l = empresaPago = pago.getReserva() != null && pago.getReserva().getEmpresa() != null ? pago.getReserva().getEmpresa().getIdEmpresa() : null;
            if (!empresaActual.equals(empresaPago)) {
                throw new IllegalArgumentException("El pago no pertenece a la empresa actual");
            }
        }
    }

    private boolean puedeVerPago(PagoReserva pago) {
        if (this.esSuperAdmin()) {
            return true;
        }
        Long empresaActual = TenantContext.requireEmpresaId();
        Long empresaPago = pago.getReserva() != null && pago.getReserva().getEmpresa() != null ? pago.getReserva().getEmpresa().getIdEmpresa() : null;
        return empresaActual.equals(empresaPago);
    }

    private boolean esSuperAdmin() {
        return TenantContext.isSuperAdmin();
    }

    private void validarMismaEmpresa(Long empresaEsperada, Long empresaRelacionada, String recurso) {
        if (empresaEsperada == null) {
            return;
        }
        if (empresaRelacionada == null || !empresaEsperada.equals(empresaRelacionada)) {
            throw new IllegalArgumentException("El recurso " + recurso + " no pertenece a la empresa actual");
        }
    }

    private String normalizarTexto(String valor) {
        if (valor == null) {
            return null;
        }
        String trimmed = valor.trim();
        return trimmed.isEmpty() ? null : trimmed;
    }

    private Reserva.EstadoReserva determinarEstadoFinanciero(BigDecimal totalReserva, BigDecimal totalPagado) {
        if (totalPagado.compareTo(BigDecimal.ZERO) <= 0) {
            return Reserva.EstadoReserva.Confirmada;
        }
        if (totalPagado.compareTo(totalReserva) >= 0) {
            return Reserva.EstadoReserva.Pagada;
        }
        return Reserva.EstadoReserva.PagoParcial;
    }

    private void validarMetodoPago(String metodo) {
        if (metodo == null || metodo.trim().isEmpty()) {
            throw new IllegalArgumentException("El m\u00e9todo de pago es obligatorio");
        }
        String metodoNormalizado = this.normalizarMetodoPago(metodo);
        if (!METODOS_VALIDOS.contains(metodoNormalizado)) {
            throw new IllegalArgumentException("M\u00e9todo de pago no soportado: " + metodo + ". M\u00e9todos v\u00e1lidos: " + String.valueOf(METODOS_VALIDOS));
        }
    }

    private String normalizarMetodoPago(String metodoPago) {
        if (metodoPago == null) {
            return null;
        }
        String trimmed = metodoPago.trim();
        if (trimmed.length() > 1) {
            return trimmed.substring(0, 1).toUpperCase() + trimmed.substring(1).toLowerCase();
        }
        return trimmed.toUpperCase();
    }

    public PagoReservaService(PagoReservaRepository pagoReservaRepository, ReservaService reservaService, CajaService cajaService, UsuarioRepository usuarioRepository) {
        this.pagoReservaRepository = pagoReservaRepository;
        this.reservaService = reservaService;
        this.cajaService = cajaService;
        this.usuarioRepository = usuarioRepository;
    }
}

