/*
 * Decompiled with CFR 0.152.
 */
package com.sistema.turistico.service;

import com.sistema.turistico.dto.CajaAperturaRequest;
import com.sistema.turistico.dto.CajaCierreRequest;
import com.sistema.turistico.dto.MovimientoCajaRequest;
import com.sistema.turistico.entity.Caja;
import com.sistema.turistico.entity.Empresa;
import com.sistema.turistico.entity.MovimientoCaja;
import com.sistema.turistico.entity.Sucursal;
import com.sistema.turistico.entity.Usuario;
import com.sistema.turistico.entity.Venta;
import com.sistema.turistico.repository.CajaRepository;
import com.sistema.turistico.repository.EmpresaRepository;
import com.sistema.turistico.repository.MovimientoCajaRepository;
import com.sistema.turistico.repository.SucursalRepository;
import com.sistema.turistico.repository.UsuarioRepository;
import com.sistema.turistico.repository.VentaRepository;
import com.sistema.turistico.security.TenantContext;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

@Service
@Transactional
public class CajaService {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(CajaService.class);
    private static final String MSG_CAJA_NO_ENCONTRADA = "Caja no encontrada";
    private final CajaRepository cajaRepository;
    private final MovimientoCajaRepository movimientoCajaRepository;
    private final EmpresaRepository empresaRepository;
    private final SucursalRepository sucursalRepository;
    private final UsuarioRepository usuarioRepository;
    private final VentaRepository ventaRepository;

    public Caja abrirCaja(CajaAperturaRequest request) {
        if (request == null) {
            throw new IllegalArgumentException("La solicitud es obligatoria");
        }
        log.info("Aperturando caja para sucursal {} y usuario {}", (Object)request.getSucursalId(), (Object)request.getUsuarioAperturaId());
        Long empresaId = TenantContext.requireEmpresaIdOrCurrent((Long)request.getEmpresaId());
        Empresa empresa = this.obtenerEmpresaActiva(empresaId);
        Sucursal sucursal = this.obtenerSucursalAutorizada(request.getSucursalId(), empresaId);
        Usuario usuarioApertura = this.obtenerUsuarioAutorizado(request.getUsuarioAperturaId(), empresaId);
        if (this.cajaRepository.existsBySucursal_IdSucursalAndEstado(sucursal.getIdSucursal(), Caja.EstadoCaja.Abierta)) {
            throw new IllegalArgumentException("Ya existe una caja abierta para la sucursal seleccionada");
        }
        Caja caja = new Caja();
        caja.setEmpresa(empresa);
        caja.setSucursal(sucursal);
        caja.setUsuarioApertura(usuarioApertura);
        caja.setFechaApertura(LocalDate.now());
        caja.setHoraApertura(LocalTime.now());
        caja.setMontoInicial(request.getMontoInicial());
        caja.setSaldoActual(request.getMontoInicial());
        caja.setObservaciones(this.normalizarTexto(request.getObservaciones()));
        caja.setEstado(Caja.EstadoCaja.Abierta);
        caja.setUsuarioCierre(null);
        caja.setMontoCierre(null);
        caja.setDiferencia(null);
        Caja cajaGuardada = (Caja)this.cajaRepository.save((Object)caja);
        log.info("Caja {} aperturada correctamente", (Object)cajaGuardada.getIdCaja());
        return cajaGuardada;
    }

    @Transactional(readOnly=true)
    public Caja obtenerCaja(Long cajaId) {
        return this.obtenerCajaAutorizada(cajaId);
    }

    @Transactional(readOnly=true)
    public Caja obtenerCajaActiva(Long cajaId) {
        Caja caja = this.obtenerCajaAutorizada(cajaId);
        if (!caja.estaAbierta()) {
            throw new IllegalStateException("La caja seleccionada no se encuentra abierta");
        }
        return caja;
    }

    @Transactional(readOnly=true)
    public List<Caja> listarCajasAbiertas(Long empresaId) {
        Long empresaFiltrada = TenantContext.requireEmpresaIdOrCurrent((Long)empresaId);
        return this.cajaRepository.findByEmpresa_IdEmpresaAndEstado(empresaFiltrada, Caja.EstadoCaja.Abierta).stream().filter(arg_0 -> this.puedeVerCaja(arg_0)).toList();
    }

    @Transactional(readOnly=true)
    public List<Caja> listarCajas(Long empresaId, Long sucursalId, Caja.EstadoCaja estado) {
        Long empresaFiltrada = TenantContext.resolveEmpresaId((Long)empresaId);
        if (sucursalId != null) {
            if (empresaFiltrada != null) {
                this.obtenerSucursalAutorizada(sucursalId, empresaFiltrada);
            } else {
                this.obtenerSucursalAutorizada(sucursalId, null);
            }
        }
        return this.cajaRepository.findByFilters(empresaFiltrada, sucursalId, estado).stream().filter(arg_0 -> this.puedeVerCaja(arg_0)).toList();
    }

    @Transactional(readOnly=true)
    public MovimientoCaja obtenerMovimiento(Long cajaId, Long movimientoId) {
        Caja caja = this.obtenerCajaAutorizada(cajaId);
        return this.movimientoCajaRepository.findByIdMovimientoAndCaja_IdCaja(movimientoId, caja.getIdCaja()).filter(movimiento -> this.puedeVerCaja(movimiento.getCaja())).orElseThrow(() -> new IllegalArgumentException("Movimiento no encontrado en la caja especificada"));
    }

    public Caja cerrarCaja(Long cajaId, CajaCierreRequest request) {
        if (request == null) {
            throw new IllegalArgumentException("La solicitud es obligatoria");
        }
        log.info("Cerrando caja {} por usuario {}", (Object)cajaId, (Object)request.getUsuarioCierreId());
        Caja caja = this.obtenerCajaAutorizada(cajaId);
        if (!caja.estaAbierta()) {
            throw new IllegalStateException("La caja ya se encontraba cerrada");
        }
        Usuario usuarioCierre = this.obtenerUsuarioAutorizado(request.getUsuarioCierreId(), caja.getEmpresa().getIdEmpresa());
        caja.setUsuarioCierre(usuarioCierre);
        caja.setMontoCierre(request.getMontoCierre());
        BigDecimal diferencia = request.getMontoCierre().subtract(caja.getSaldoActual());
        caja.setDiferencia(diferencia);
        String observacion = this.normalizarTexto(request.getObservaciones());
        if (observacion != null) {
            caja.setObservaciones(observacion);
        }
        caja.setEstado(Caja.EstadoCaja.Cerrada);
        Caja cajaCerrada = (Caja)this.cajaRepository.save((Object)caja);
        log.info("Caja {} cerrada. Diferencia registrada: {}", (Object)cajaId, (Object)diferencia);
        return cajaCerrada;
    }

    public MovimientoCaja registrarMovimiento(Long cajaId, MovimientoCajaRequest request) {
        if (request == null) {
            throw new IllegalArgumentException("La solicitud es obligatoria");
        }
        Caja caja = this.obtenerCajaActiva(cajaId);
        MovimientoCaja movimiento = new MovimientoCaja();
        movimiento.setCaja(caja);
        movimiento.setTipoMovimiento(request.getTipoMovimiento());
        movimiento.setMonto(request.getMonto());
        movimiento.setDescripcion(this.normalizarTexto(request.getDescripcion()));
        movimiento.setFechaHora(LocalDateTime.now());
        if (request.getVentaId() != null) {
            Venta venta = (Venta)this.ventaRepository.findById((Object)request.getVentaId()).orElseThrow(() -> new IllegalArgumentException("Venta no encontrada"));
            this.validarMismaEmpresa(caja.getEmpresa().getIdEmpresa(), venta.getEmpresa() != null ? venta.getEmpresa().getIdEmpresa() : null, "venta");
            if (venta.getCaja() != null && !venta.getCaja().getIdCaja().equals(caja.getIdCaja())) {
                throw new IllegalArgumentException("La venta no pertenece a la caja seleccionada");
            }
            movimiento.setVenta(venta);
        }
        BigDecimal nuevoSaldo = this.calcularNuevoSaldo(caja.getSaldoActual(), request.getTipoMovimiento(), request.getMonto());
        caja.setSaldoActual(nuevoSaldo);
        MovimientoCaja movimientoGuardado = (MovimientoCaja)this.movimientoCajaRepository.save((Object)movimiento);
        this.cajaRepository.save((Object)caja);
        log.info("Movimiento {} registrado para caja {}", (Object)movimientoGuardado.getIdMovimiento(), (Object)cajaId);
        return movimientoGuardado;
    }

    @Transactional(readOnly=true)
    public List<MovimientoCaja> obtenerMovimientos(Long cajaId, LocalDateTime inicio, LocalDateTime fin, MovimientoCaja.TipoMovimiento tipo) {
        Caja caja = this.obtenerCajaAutorizada(cajaId);
        return this.movimientoCajaRepository.findByCajaAndFilters(caja.getIdCaja(), tipo, inicio, fin);
    }

    @Transactional(readOnly=true)
    public Map<String, Object> obtenerArqueo(Long cajaId, LocalDate fechaInicio, LocalDate fechaFin) {
        Caja caja = this.obtenerCajaAutorizada(cajaId);
        LocalDateTime inicio = fechaInicio != null ? fechaInicio.atStartOfDay() : null;
        LocalDateTime fin = fechaFin != null ? fechaFin.atTime(LocalTime.MAX) : null;
        BigDecimal ingresos = this.movimientoCajaRepository.sumByCajaAndTipoAndRango(caja.getIdCaja(), MovimientoCaja.TipoMovimiento.Ingreso, inicio, fin);
        BigDecimal egresos = this.movimientoCajaRepository.sumByCajaAndTipoAndRango(caja.getIdCaja(), MovimientoCaja.TipoMovimiento.Egreso, inicio, fin);
        HashMap<String, Object> resumen = new HashMap<String, Object>();
        resumen.put("cajaId", caja.getIdCaja());
        resumen.put("fechaApertura", caja.getFechaApertura());
        resumen.put("saldoInicial", caja.getMontoInicial());
        resumen.put("saldoActual", caja.getSaldoActual());
        resumen.put("ingresos", ingresos);
        resumen.put("egresos", egresos);
        resumen.put("saldoCalculado", caja.getMontoInicial().add(ingresos).subtract(egresos));
        return resumen;
    }

    public MovimientoCaja anularMovimiento(Long cajaId, Long movimientoId, String motivo) {
        log.info("Anulando movimiento {} de caja {}", (Object)movimientoId, (Object)cajaId);
        Caja caja = this.obtenerCajaActiva(cajaId);
        MovimientoCaja movimiento = (MovimientoCaja)this.movimientoCajaRepository.findById((Object)movimientoId).orElseThrow(() -> new IllegalArgumentException("Movimiento no encontrado"));
        if (!movimiento.getCaja().getIdCaja().equals(caja.getIdCaja())) {
            throw new IllegalArgumentException("El movimiento no pertenece a la caja especificada");
        }
        if (movimiento.getVenta() != null) {
            throw new IllegalArgumentException("No se pueden anular movimientos autom\u00e1ticos generados por ventas");
        }
        MovimientoCaja movimientoReverso = new MovimientoCaja();
        movimientoReverso.setCaja(caja);
        movimientoReverso.setTipoMovimiento(MovimientoCaja.TipoMovimiento.Ingreso.equals((Object)movimiento.getTipoMovimiento()) ? MovimientoCaja.TipoMovimiento.Egreso : MovimientoCaja.TipoMovimiento.Ingreso);
        movimientoReverso.setMonto(movimiento.getMonto());
        String descripcion = movimiento.getDescripcion() != null ? movimiento.getDescripcion() : "";
        movimientoReverso.setDescripcion("ANULACI\u00d3N: " + descripcion + (String)(motivo != null && !motivo.isBlank() ? " - Motivo: " + motivo.trim() : ""));
        movimientoReverso.setFechaHora(LocalDateTime.now());
        BigDecimal nuevoSaldo = this.calcularNuevoSaldo(caja.getSaldoActual(), movimientoReverso.getTipoMovimiento(), movimientoReverso.getMonto());
        caja.setSaldoActual(nuevoSaldo);
        MovimientoCaja movimientoGuardado = (MovimientoCaja)this.movimientoCajaRepository.save((Object)movimientoReverso);
        this.cajaRepository.save((Object)caja);
        log.info("Movimiento {} anulado exitosamente con reverso {}", (Object)movimientoId, (Object)movimientoGuardado.getIdMovimiento());
        return movimientoGuardado;
    }

    private Caja obtenerCajaAutorizada(Long cajaId) {
        Caja caja = (Caja)this.cajaRepository.findById((Object)cajaId).orElseThrow(() -> new IllegalArgumentException(MSG_CAJA_NO_ENCONTRADA));
        this.validarPertenencia(caja);
        return caja;
    }

    private Usuario obtenerUsuarioAutorizado(Long usuarioId, Long empresaId) {
        Long empresaActual;
        Long resolvedId = usuarioId != null ? usuarioId : TenantContext.requireUserId();
        Usuario usuario = (Usuario)this.usuarioRepository.findActivoById(resolvedId).orElseThrow(() -> new IllegalArgumentException("Usuario no encontrado"));
        Long empresaUsuario = usuario.getEmpresa() != null ? usuario.getEmpresa().getIdEmpresa() : null;
        this.validarMismaEmpresa(empresaId, empresaUsuario, "usuario");
        if (!this.esSuperAdmin() && !(empresaActual = TenantContext.requireEmpresaId()).equals(empresaUsuario)) {
            throw new IllegalArgumentException("El usuario no pertenece a la empresa actual");
        }
        return usuario;
    }

    private Sucursal obtenerSucursalAutorizada(Long sucursalId, Long empresaId) {
        Long empresaActual;
        if (sucursalId == null) {
            throw new IllegalArgumentException("La sucursal es obligatoria");
        }
        Sucursal sucursal = (Sucursal)this.sucursalRepository.findActivaById(sucursalId).orElseThrow(() -> new IllegalArgumentException("Sucursal no encontrada"));
        Long empresaSucursal = sucursal.getEmpresa() != null ? sucursal.getEmpresa().getIdEmpresa() : null;
        this.validarMismaEmpresa(empresaId, empresaSucursal, "sucursal");
        if (!this.esSuperAdmin() && !(empresaActual = TenantContext.requireEmpresaId()).equals(empresaSucursal)) {
            throw new IllegalArgumentException("La sucursal no pertenece a la empresa actual");
        }
        return sucursal;
    }

    private Empresa obtenerEmpresaActiva(Long empresaId) {
        return (Empresa)this.empresaRepository.findActivaById(empresaId).orElseThrow(() -> new IllegalArgumentException("Empresa no encontrada"));
    }

    private void validarPertenencia(Caja caja) {
        if (!this.esSuperAdmin()) {
            Long empresaCaja;
            Long empresaActual = TenantContext.requireEmpresaId();
            Long l = empresaCaja = caja.getEmpresa() != null ? caja.getEmpresa().getIdEmpresa() : null;
            if (!empresaActual.equals(empresaCaja)) {
                throw new IllegalArgumentException("La caja no pertenece a la empresa actual");
            }
        }
    }

    private boolean puedeVerCaja(Caja caja) {
        if (this.esSuperAdmin()) {
            return true;
        }
        Long empresaActual = TenantContext.requireEmpresaId();
        Long empresaCaja = caja.getEmpresa() != null ? caja.getEmpresa().getIdEmpresa() : null;
        return empresaActual.equals(empresaCaja);
    }

    private boolean esSuperAdmin() {
        return TenantContext.isSuperAdmin();
    }

    private void validarMismaEmpresa(Long empresaEsperada, Long empresaRelacionada, String recurso) {
        if (empresaEsperada == null) {
            return;
        }
        if (empresaRelacionada == null || !empresaEsperada.equals(empresaRelacionada)) {
            throw new IllegalArgumentException("El " + recurso + " no pertenece a la empresa actual");
        }
    }

    private BigDecimal calcularNuevoSaldo(BigDecimal saldoActual, MovimientoCaja.TipoMovimiento tipo, BigDecimal monto) {
        BigDecimal resultado;
        BigDecimal bigDecimal = resultado = MovimientoCaja.TipoMovimiento.Ingreso.equals((Object)tipo) ? saldoActual.add(monto) : saldoActual.subtract(monto);
        if (resultado.compareTo(BigDecimal.ZERO) < 0) {
            throw new IllegalStateException("El saldo de caja no puede ser negativo");
        }
        return resultado;
    }

    private String normalizarTexto(String valor) {
        if (valor == null) {
            return null;
        }
        String trimmed = valor.trim();
        return trimmed.isEmpty() ? null : trimmed;
    }

    @Generated
    public CajaService(CajaRepository cajaRepository, MovimientoCajaRepository movimientoCajaRepository, EmpresaRepository empresaRepository, SucursalRepository sucursalRepository, UsuarioRepository usuarioRepository, VentaRepository ventaRepository) {
        this.cajaRepository = cajaRepository;
        this.movimientoCajaRepository = movimientoCajaRepository;
        this.empresaRepository = empresaRepository;
        this.sucursalRepository = sucursalRepository;
        this.usuarioRepository = usuarioRepository;
        this.ventaRepository = ventaRepository;
    }
}

